#include <lemon/list_graph.h>
#include <lemon/network_simplex.h>
#include <lemon/cost_scaling.h>
#include <lemon/capacity_scaling.h>
#include <lemon/cycle_canceling.h>
#include <sys/time.h>

#include <cstdio>
#include <vector>
#include <iostream>
#include <cassert>


template <class ConstraintIterator>
void solve( int n, int m, 
       ConstraintIterator begin ) {


    std::vector<lemon::ListDigraph::Node> nodes(n);

    lemon::ListDigraph g;
    lemon::ListDigraph::ArcMap<long long int> cap(g);
    lemon::ListDigraph::ArcMap<long long int> wgt(g);

    std::cout << " INFO: adding nodes. " << std::endl;
    for ( int i = 0; i < n; ++i ) {
        nodes[i] = g.addNode();
    }

    std::cout << " INFO: adding arcs. " << std::endl;
    std::vector<lemon::ListDigraph::Arc> arcs(2*m);
    ConstraintIterator cit = begin;
    for ( int iarc = 0; iarc < m; ++iarc, ++cit ) {
        int s = cit->s; int t = cit->t; 
        assert(s >= 0 );
        assert(s < n );
        assert(t >= 0);
        assert(t < n);
        arcs[2*iarc+0] = g.addArc(nodes[s],nodes[t]);
        arcs[2*iarc+1] = g.addArc(nodes[t],nodes[s]);
        int b = cit->b;
        int c = cit->c;
        wgt[arcs[2*iarc+0]] = b;
        wgt[arcs[2*iarc+1]] = -b;
        cap[arcs[2*iarc+0]] = c;
        cap[arcs[2*iarc+1]] = c;
    }


    {
        lemon::CapacityScaling<lemon::ListDigraph> cs(g);
        std::cout << " INFO: running capacity scaling. " << std::endl;
        struct timeval t0, t1;
        gettimeofday(&t0, 0);
        lemon::CapacityScaling<lemon::ListDigraph>::ProblemType csresult =
            cs.upperMap(cap).costMap(wgt).run();
        gettimeofday(&t1, 0);
        long long elapsed = (t1.tv_sec-t0.tv_sec)*1000000LL + t1.tv_usec-t0.tv_usec;
        double sec = (double) elapsed /  1000000.0;
        std::cout << " TIME: capacity scaling: " << sec << "s" << std::endl;
        switch ( csresult ) {
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::INFEASIBLE:
                std::cout << " INFO: solution is INFEASIBLE" << std::endl;
                break;
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::OPTIMAL:
                std::cout << " INFO: solution is OPTIMAL" << std::endl;
                break;
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::UNBOUNDED:
                std::cout << " INFO: solution is UNBOUNDED" << std::endl;
                break;
            default:
                std::cout << " INFO: not understood return of capacity scaling" << std::endl;
                break;
        }
	int totalcost = cs.totalCost();
	std::cout << " INFO: cost scaling: " << totalcost << std::endl;
    }

    {
        lemon::CostScaling<lemon::ListDigraph> cs(g);
        std::cout << " INFO: running cost scaling. " << std::endl;
        struct timeval t0, t1;
        gettimeofday(&t0, 0);
        lemon::CostScaling<lemon::ListDigraph>::ProblemType csresult =
            cs.upperMap(cap).costMap(wgt).run();
        gettimeofday(&t1, 0);
        long long elapsed = (t1.tv_sec-t0.tv_sec)*1000000LL + t1.tv_usec-t0.tv_usec;
        double sec = (double) elapsed /  1000000.0;
        std::cout << " TIME: cost scaling: " << sec << "s" << std::endl;
        switch ( csresult ) {
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::INFEASIBLE:
                std::cout << " INFO: solution is INFEASIBLE" << std::endl;
                break;
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::OPTIMAL:
                std::cout << " INFO: solution is OPTIMAL" << std::endl;
                break;
            case lemon::CostScaling<lemon::ListDigraph>::ProblemType::UNBOUNDED:
                std::cout << " INFO: solution is UNBOUNDED" << std::endl;
                break;
            default:
                std::cout << " INFO: not understood return of cost scaling" << std::endl;
                break;
        }
	int totalcost = cs.totalCost();
	std::cout << " INFO: cost scaling: " << totalcost << std::endl;
    }

    {
    lemon::NetworkSimplex<lemon::ListDigraph> ns(g);
    struct timeval t0, t1;
    gettimeofday(&t0, 0);
    lemon::NetworkSimplex<lemon::ListDigraph>::ProblemType result =
        ns.upperMap(cap).costMap(wgt).run();
    gettimeofday(&t1, 0);
    long long elapsed = (t1.tv_sec-t0.tv_sec)*1000000LL + t1.tv_usec-t0.tv_usec;
    double sec = (double) elapsed /  1000000.0;
    std::cout << " TIME: network simplex: " << sec << "s" << std::endl;

    switch ( result ) {
        case lemon::NetworkSimplex<lemon::ListDigraph>::ProblemType::INFEASIBLE:
            std::cout << " INFO: solution is INFEASIBLE" << std::endl;
            break;
        case lemon::NetworkSimplex<lemon::ListDigraph>::ProblemType::OPTIMAL:
            std::cout << " INFO: solution is OPTIMAL" << std::endl;
            break;
        case lemon::NetworkSimplex<lemon::ListDigraph>::ProblemType::UNBOUNDED:
            std::cout << " INFO: solution is UNBOUNDED" << std::endl;
            break;
        default:
            std::cout << " INFO: not understood return of network simplex" << std::endl;
            break;
    }

    int totalcost = ns.totalCost();
    std::cout << " INFO: network simplex cost: " << totalcost << std::endl;
    }

    return; 

}
